'use strict';

const fs = require('fs');
const url = require('url');
const os = require('os');
const resolvePath = require('path').resolve;
const isAbsolutePath = require('path-is-absolute');

const promises = require('../promises');


exports.Files = Files;
exports.uriToPath = uriToPath;

const readFile = promises.promisify(fs.readFile.bind(fs));

function Files(base) {
  function read(uri, encoding) {
    return resolveUri(uri).then(function(path) {
      return readFile(path, encoding).caught(function(error) {
        const message = "could not open external image: '" + uri + "' (document directory: '" + base + "')\n" + error.message;
        return promises.reject(new Error(message));
      });
    });
  }

  function resolveUri(uri) {
    const path = uriToPath(uri);
    if (isAbsolutePath(path)) {
      return promises.resolve(path);
    } else if (base) {
      return promises.resolve(resolvePath(base, path));
    }
    return promises.reject(new Error("could not find external image '" + uri + "', path of input document is unknown"));

  }

  return {
    read,
  };
}

function uriToPath(uriString, platform) {
  if (!platform) {
    platform = os.platform();
  }

  const uri = url.parse(uriString);
  if (isLocalFileUri(uri) || isRelativeUri(uri)) {
    const path = decodeURIComponent(uri.path);
    if (platform === 'win32' && /^\/[a-z]:/i.test(path)) {
      return path.slice(1);
    }
    return path;

  }
  throw new Error('Could not convert URI to path: ' + uriString);
}

function isLocalFileUri(uri) {
  return uri.protocol === 'file:' && (!uri.host || uri.host === 'localhost');
}

function isRelativeUri(uri) {
  return !uri.protocol && !uri.host;
}
